<?php
/**
 * Creative Elements - Elementor based PageBuilder
 *
 * @author    WebshopWorks, Elementor
 * @copyright 2019-2020 WebshopWorks.com & Elementor.com
 * @license   https://www.gnu.org/licenses/gpl-3.0.html
 */

namespace CE;

defined('_PS_VERSION_') or die;

class Frontend
{
    private $_enqueue_google_fonts = array();
    private $_enqueue_google_early_access_fonts = array();

    private $_is_frontend_mode = false;
    private $_has_elementor_in_page = false;

    /**
     * @var Stylesheet
     */
    private $stylesheet;

    public function init()
    {
        $this->_is_frontend_mode = true;
        $this->_has_elementor_in_page = Plugin::instance()->db->hasElementorInPost(get_the_ID());

        $this->_initStylesheet();

        add_filter('elementor/frontend/the_content', array('\CE\Helper', 'filterTheContent'));

        // add_action('wp_head', array($this, 'print_css'));
        // add_filter('body_class', array($this, 'body_class'));

        // add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
        // add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));

        // add_action('wp_footer', array($this, 'wp_footer'));
    }

    private function _initStylesheet()
    {
        $this->stylesheet = new Stylesheet();

        $breakpoints = Responsive::getBreakpoints();

        $this->stylesheet
            ->addDevice('mobile', $breakpoints['md'] - 1)
            ->addDevice('tablet', $breakpoints['lg'] - 1);
    }

    protected function _printElements($elements_data)
    {
        foreach ($elements_data as $element_data) {
            $element = Plugin::instance()->elements_manager->createElementInstance($element_data);

            $element->printElement();
        }
    }

    public function enqueueScripts()
    {
        do_action('elementor/frontend/before_enqueue_scripts');

        $suffix = defined('_PS_MODE_DEV_') && _PS_MODE_DEV_ ? '' : '.min';

        wp_register_script(
            'waypoints',
            _CE_ASSETS_URL_ . 'lib/waypoints/waypoints' . $suffix . '.js',
            array(
                'jquery',
            ),
            '4.0.1',
            true
        );

        wp_register_script(
            'jquery-numerator',
            _CE_ASSETS_URL_ . 'lib/jquery-numerator/jquery-numerator' . $suffix . '.js',
            array(
                'jquery',
            ),
            '0.2.1',
            true
        );

        wp_register_script(
            'jquery-slick',
            _CE_ASSETS_URL_ . 'lib/slick/slick' . $suffix . '.js',
            array(
                'jquery',
            ),
            '1.6.1',
            true
        );

        wp_register_script(
            'elementor-frontend',
            _CE_ASSETS_URL_ . 'js/frontend' . $suffix . '.js',
            array(
                'waypoints',
                'jquery-numerator',
                'jquery-slick',
            ),
            _CE_VERSION_,
            true
        );
        wp_enqueue_script('elementor-frontend');

        wp_localize_script(
            'elementor-frontend',
            'ceFrontendConfig',
            array(
                'isEditMode' => Plugin::instance()->editor->isEditMode(),
                'stretchedSectionContainer' => get_option('elementor_stretched_section_container', ''),
                'is_rtl' => ($lang = \Language::getLanguage(\Tools::getValue('id_lang'))) && !empty($lang['is_rtl']),
            )
        );
    }

    public function enqueueStyles()
    {
        $suffix = defined('_PS_MODE_DEV_') && _PS_MODE_DEV_ ? '' : '.min';

        $direction_suffix = is_rtl() ? '-rtl' : '';

        wp_enqueue_style(
            'elementor-icons',
            _CE_ASSETS_URL_ . 'lib/eicons/css/elementor-icons' . $suffix . '.css',
            array(),
            _CE_VERSION_
        );

        wp_register_style(
            'font-awesome',
            _CE_ASSETS_URL_ . 'lib/font-awesome/css/font-awesome' . $suffix . '.css',
            array(),
            '4.7.0'
        );

        // Elementor Animations
        wp_register_style(
            'elementor-animations',
            _CE_ASSETS_URL_ . 'css/animations.min.css',
            array(),
            _CE_VERSION_
        );

        wp_register_style(
            'elementor-frontend',
            _CE_ASSETS_URL_ . 'css/frontend' . $direction_suffix . $suffix . '.css',
            array(),
            _CE_VERSION_
        );

        wp_enqueue_style('elementor-icons');
        wp_enqueue_style('font-awesome');
        wp_enqueue_style('elementor-animations');
        wp_enqueue_style('elementor-frontend');

        $css_file = new Post_CSS_File(get_the_ID());
        $css_file->enqueue();
    }

    public function printCss()
    {
        $container_width = absint(get_option('elementor_container_width'));
        if (!empty($container_width)) {
            $this->stylesheet->addRules('.elementor-section.elementor-section-boxed > .elementor-container', 'max-width:' . $container_width . 'px');
        }

        $this->_parseSchemesCssCode();

        $css_code = $this->stylesheet;

        if (empty($css_code)) {
            return;
        }

        \CESmarty::call(_CE_TEMPLATES_ . 'front/frontend.tpl', 'Frontend_printCss', array('css_code' => $css_code));

        $this->printGoogleFonts();
    }

    /**
     * Handle style that do not printed in header
     */
    public function wpFooter()
    {
        // TODO: add JS to append the css to the `head` tag
        $this->printGoogleFonts();
    }

    public function printGoogleFonts()
    {
        // Enqueue used fonts
        if (!empty($this->_enqueue_google_fonts)) {
            foreach ($this->_enqueue_google_fonts as &$font) {
                $font = str_replace(' ', '+', $font) . ':100,100italic,200,200italic,300,300italic,400,400italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic';
            }

            $fonts_url = sprintf('https://fonts.googleapis.com/css?family=%s', implode('|', $this->_enqueue_google_fonts));

            $subsets = array(
                'ru' => 'cyrillic',
                'bg' => 'cyrillic',
                'he' => 'hebrew',
                'el' => 'greek',
                'vi' => 'vietnamese',
                'uk' => 'cyrillic',
            );
            $locale = \Context::getContext()->language->iso_code;

            if (isset($subsets[$locale])) {
                $fonts_url .= '&subset=' . $subsets[$locale];
            }

            $this->_enqueue_google_fonts = array();
        }

        \CESmarty::call(_CE_TEMPLATES_ . 'front/frontend.tpl', 'Frontend_printGoogleFonts', array(
            'fonts_url' => isset($fonts_url) ? $fonts_url : '',
            'early_access_fonts' => &$this->_enqueue_google_early_access_fonts,
        ));

        $this->_enqueue_google_early_access_fonts = array();
    }

    public function addEnqueueFont($font)
    {
        switch (Fonts::getFontType($font)) {
            case Fonts::GOOGLE:
                if (!in_array($font, $this->_enqueue_google_fonts)) {
                    $this->_enqueue_google_fonts[] = $font;
                }
                break;

            case Fonts::EARLYACCESS:
                if (!in_array($font, $this->_enqueue_google_early_access_fonts)) {
                    $this->_enqueue_google_early_access_fonts[] = $font;
                }
                break;
        }
    }

    protected function _parseSchemesCssCode()
    {
        foreach (Plugin::instance()->widgets_manager->getWidgetTypes() as $widget) {
            $scheme_controls = $widget->getSchemeControls();

            foreach ($scheme_controls as $control) {
                PostCssFile::addControlRules($this->stylesheet, $control, $widget->getControls(), function ($control) {
                    $scheme_value = Plugin::instance()->schemes_manager->getSchemeValue($control['scheme']['type'], $control['scheme']['value']);

                    if (empty($scheme_value)) {
                        return null;
                    }

                    if (!empty($control['scheme']['key'])) {
                        $scheme_value = $scheme_value[$control['scheme']['key']];
                    }

                    if (empty($scheme_value)) {
                        return null;
                    }

                    $control_obj = Plugin::instance()->controls_manager->getControl($control['type']);

                    if (ControlsManager::FONT === $control_obj->getType()) {
                        ${'this'}->addEnqueueFont($scheme_value);
                    }

                    return $scheme_value;
                }, array('{{WRAPPER}}'), array('.elementor-widget-' . $widget->getName()));
            }
        }
    }

    public function applyBuilderInContent($content)
    {
        if (!$this->_is_frontend_mode) {
            return $content;
        }

        $post_id = get_the_ID();
        $builder_content = $this->getBuilderContent($post_id);

        if (!empty($builder_content)) {
            $content = $builder_content;
        }

        // Add the filter again for other `the_content` calls
        // add_filter('the_content', array($this, 'apply_builder_in_content'));

        return $content;
    }

    public function getBuilderContent($post_id, $with_css = false)
    {
        $edit_mode = Plugin::instance()->db->getEditMode($post_id);

        if ('builder' !== $edit_mode) {
            return '';
        }

        $data = Plugin::instance()->db->getPlainEditor($post_id);
        $data = apply_filters('elementor/frontend/builder_content_data', $data, $post_id);

        if (empty($data)) {
            return '';
        }

        $css_file = new PostCssFile($post_id);
        $css_file->enqueue();

        ob_start();
        \CESmarty::call(_CE_TEMPLATES_ . 'front/frontend.tpl', 'Frontend_getBuilderContent', array(
            'css_file' => $with_css ? $css_file : null,
            'post_id' => $post_id,
        ));
        $this->_printElements($data);
        \CESmarty::write(_CE_TEMPLATES_ . 'front/frontend.tpl', 'ElementBase_afterRender');

        return apply_filters('elementor/frontend/the_content', ob_get_clean());
    }

    public function getBuilderContentForDisplay($post_id)
    {
        if (!get_post($post_id)) {
            return '';
        }

        // Avoid recursion
        if (get_the_ID() === (int) $post_id) {
            $content = '';
            if (Plugin::instance()->editor->isEditMode()) {
                $msg = __('Invalid Data: The Template ID cannot be the same as the currently edited template. Please choose a different one.', 'elementor');
                $content = \CESmarty::sprintf(_CE_TEMPLATES_ . 'front/frontend.tpl', 'Frontend_getBuilderContentForDisplay', $msg);
            }

            return $content;
        }

        // Set edit mode as false, so don't render settings and etc. use the $is_edit_mode to indicate if we need the css inline
        $is_edit_mode = Plugin::instance()->editor->isEditMode();
        Plugin::instance()->editor->setEditMode(false);

        // Change the global post to current library post, so widgets can use `get_the_ID` and other post data
        // if (isset($GLOBALS['post'])) {
        //     $global_post = $GLOBALS['post'];
        // }

        // $GLOBALS['post'] = get_post($post_id);

        $content = $this->getBuilderContent($post_id, $is_edit_mode);

        if (!empty($content)) {
            $this->_hasElementorInPage = true;
        }

        // Restore global post
        // if (isset($global_post)) {
        //     $GLOBALS['post'] = $global_post;
        // } else {
        //     unset($GLOBALS['post']);
        // }

        // Restore edit mode state
        Plugin::instance()->editor->setEditMode($is_edit_mode);

        return $content;
    }

    public function __construct()
    {
        // We don't need this class in admin side, but in AJAX requests
        if (is_admin() && !\Tools::getIsset('ajax')) {
            return;
        }
        $this->init();

        // add_action( 'template_redirect', [ $this, 'init' ] );
        add_filter('the_content', array($this, 'apply_builder_in_content'));
    }
}
