<?php
/**
 * Creative Elements - Elementor based PageBuilder
 *
 * @author    WebshopWorks
 * @copyright 2019-2020 WebshopWorks.com
 * @license   One domain support license
 */

namespace CE;

defined('_PS_VERSION_') or die;

require_once _CE_PATH_ . 'classes/wrappers/UId.php';
require_once _CE_PATH_ . 'classes/wrappers/Post.php';

class Helper
{
    public static $actions = array();

    public static $filters = array();

    public static $styles = array();

    public static $scripts = array();

    public static $head_styles = array();

    public static $head_scripts = array();

    public static $body_scripts = array();

    public static $enqueue_css = array();

    public static $productHooks = array(
        'displayfooterproduct',
        'displayproductadditionalinfo',
        'displayproductlistreviews',
        'displayproductpriceblock',
        'displayafterproductthumbs',
        'displayleftcolumnproduct',
        'displayrightcolumnproduct',
    );

    public static function getAjaxLink()
    {
        return preg_replace('/\buid=\d+(&footerProduct=\d+)?/', 'ajax=1', $_SERVER['REQUEST_URI']);
    }

    public static function getWrapfix()
    {
        $wrapfix = array(
            'classic' => version_compare(_PS_VERSION_, '1.7.6', '<') ? 'featured-products wrapfix' : '',
            'classic-rocket' => 'wrapfix',
        );
        return isset($wrapfix[_THEME_NAME_]) ? $wrapfix[_THEME_NAME_] : '';
    }

    public static function getMediaLink($url, $full = false)
    {
        if ($url && !preg_match('~^(https?:)?//~i', $url)) {
            $url = __PS_BASE_URI__ . $url;

            if (_MEDIA_SERVER_1_ || $full) {
                $url = \Context::getContext()->link->getMediaLink($url);
            }
        }
        return $url;
    }

    public static function getPageLayouts()
    {
        $context = \Context::getContext();

        if (!empty($context->shop->theme)) {
            $theme_settings = $context->shop->theme->get('theme_settings');

            return $theme_settings['layouts'];
        }
        $layouts = array();
        $prefix = _DB_PREFIX_;
        $id_theme = (int) $context->shop->id;
        $pages = \Db::getInstance()->executeS(
            "SELECT m.`page`, tm.`left_column`, tm.`right_column`
            FROM {$prefix}theme_meta as tm
            INNER JOIN {$prefix}meta m ON m.`id_meta` = tm.`id_meta`
            WHERE tm.`id_theme` = $id_theme"
        );
        if (!empty($pages)) {
            $map = array(
                '00' => 'layout-one-column',
                '01' => 'layout-right-column',
                '10' => 'layout-left-column',
                '11' => 'layout-both-columns',
            );
            foreach ($pages as &$page) {
                $layouts[$page['page']] = $map[$page['left_column'] . $page['right_column']];
            }
        }

        return $layouts;
    }

    public static function filterTheContent($content)
    {
        static $editor = null;

        // Edit with CE
        if (null === $editor && $editor = self::getEditorLink()) {
            $content = \CESmarty::capture(_CE_TEMPLATES_ . 'admin/admin.tpl', 'ce_render', array('elems' => array(
                'script' => array(
                    'attrs' => array(
                        'src' => _CE_ASSETS_URL_ . 'js/frontend-edit.js?v=' . _CE_VERSION_,
                        'data-ce-editor' => $editor,
                        'data-ce-title' => __('Edit with Creative Elements', 'elementor'),
                    ),
                    'children' => '',
                ),
            ))) . $content;

            $href = esc_attr(_CE_ASSETS_URL_ . 'css/frontend-edit.css?v=' . _CE_VERSION_);
            Helper::$enqueue_css[] = \CESmarty::sprintf(_CE_TEMPLATES_ . 'admin/admin.tpl', 'ce_stylesheet', $href, 'all');
        }
        return self::getCleanCss() . $content;
    }

    private static function getCleanCss()
    {
        static $global;
        $css = '';

        if (null === $global) {
            $global = true;

            ob_start();
            Plugin::instance()->frontend->printCss();
            $css = ob_get_clean();
        }
        while (!empty(Helper::$enqueue_css)) {
            $css .= array_pop(Helper::$enqueue_css);
        }
        return $css;
    }

    private static function getEditorLink()
    {
        static $link;

        if (null === $link) {
            $link = '';

            if (\Configuration::get('elementor_frontend_edit') &&
                ($id_employee = self::getEmployeeId()) &&
                ($dir = glob(_PS_ROOT_DIR_ . '/*/filemanager', GLOB_ONLYDIR))
            ) {
                $tab = 'AdminCEEditor';
                $link = __PS_BASE_URI__ . basename(dirname($dir[0])) . '/index.php?' . http_build_query(array(
                    'controller' => $tab,
                    'token' => \Tools::getAdminToken($tab . (int) \Tab::getIdFromClassName($tab) . $id_employee),
                ));
            }
        }
        return $link;
    }

    private static function getEmployeeId()
    {
        static $id_employee;

        if (null === $id_employee) {
            $lifetime = max((int) \Configuration::get('PS_COOKIE_LIFETIME_BO'), 1);
            $cookie = new \Cookie('psAdmin', '', time() + $lifetime * 3600);
            $id_employee = !empty($cookie->id_employee) ? (int) $cookie->id_employee : 0;
        }
        return $id_employee;
    }

    public static function transError($error)
    {
        return _CE_PS16_
            ? \Tools::displayError($error, false)
            : \Context::getContext()->getTranslator()->trans($error, array(), 'Admin.Notifications.Error')
        ;
    }

    public static function remoteRender(array $data)
    {
        if (!$uid = get_the_ID()) {
            return;
        }
        $url = get_preview_post_link($uid, array(), false);

        empty($uid) or $res = wp_remote_post($url, array(
            'timeout' => 5,
            'body' => array(
                'render' => 'widget',
                'edit' => 0,
                'data' => json_encode($data),
            ),
        ));
        empty($res) or $resp = json_decode($res, true);

        echo isset($resp['data']['render']) ? \Tools::substr(trim($resp['data']['render']), 40, -6) : '';
    }
}

function add_action($tag, $callback, $priority = 10, $accepted_args = 1)
{
    $p = (int) $priority;

    if (!isset(Helper::$actions[$tag])) {
        Helper::$actions[$tag] = array();
    }
    if (!isset(Helper::$actions[$tag][$p])) {
        Helper::$actions[$tag][$p] = array();
    }
    if (is_string($callback) && '\\' != $callback[0]) {
        $callback = '\\' . __NAMESPACE__ . '\\' . $callback;
    }
    if (is_array($callback) && !empty($callback[1]) && stripos($callback[1], '_') !== false) {
        $callback[1] = \Tools::toCamelCase($callback[1]);
    }
    Helper::$actions[$tag][$p][] = $callback;
}

function do_action($tag, $arg = '')
{
    if (isset(Helper::$actions[$tag])) {
        $actions = &Helper::$actions[$tag];

        $args = func_get_args();
        array_shift($args);

        $priorities = array_keys($actions);
        sort($priorities, SORT_NUMERIC);

        foreach ($priorities as $p) {
            foreach ($actions[$p] as $callback) {
                call_user_func_array($callback, $args);
            }
        }
    }
}

function do_action_ref_array($tag, array $args = array())
{
    array_unshift($args, $tag);
    call_user_func_array(__NAMESPACE__ . '\do_action', $args);
}

function add_filter($tag, $callback, $priority = 10, $accepted_args = 1)
{
    $p = (int) $priority;

    if (!isset(Helper::$filters[$tag])) {
        Helper::$filters[$tag] = [];
    }
    if (!isset(Helper::$filters[$tag][$p])) {
        Helper::$filters[$tag][$p] = [];
    }
    if (is_string($callback) && '\\' != $callback[0]) {
        $callback = '\\' . __NAMESPACE__ . '\\' . $callback;
    }
    if (is_array($callback) && !empty($callback[1]) && stripos($callback[1], '_') !== false) {
        $callback[1] = \Tools::toCamelCase($callback[1]);
    }
    Helper::$filters[$tag][$p][] = $callback;
}

function has_filter($tag, $function_to_check = false)
{
    if ($function_to_check) {
        throw new \RuntimeException('TODO');
    }
    return isset(Helper::$filters[$tag]);
}

function apply_filters($tag, $value)
{
    if (isset(Helper::$filters[$tag])) {
        $filters = &Helper::$filters[$tag];

        $args = func_get_args();
        array_shift($args);

        $priorities = array_keys($filters);
        sort($priorities, SORT_NUMERIC);

        foreach ($priorities as $p) {
            foreach ($filters[$p] as $callback) {
                $args[0] = call_user_func_array($callback, $args);
            }
        }
        return $args[0];
    }
    return $value;
}

function clean($cnt)
{
    return $cnt;
}

add_action('smarty/before_fetch', function ($smarty) {
    $GLOBALS['_esc'] = $smarty->escape_html;
    $smarty->escape_html = false;
    $smarty->registered_plugins['modifier']['cleanHtml'][0] = 'CE\clean';
}, 0);

add_action('smarty/after_fetch', function ($smarty) {
    $smarty->escape_html = $GLOBALS['_esc'];
    $smarty->registered_plugins['modifier']['cleanHtml'][0] = 'smartyCleanHtml';
}, 0);

add_action('smarty/before_call', function ($smarty) {
    $smarty->registered_plugins['modifier']['cleanHtml'][0] = 'CE\clean';
}, 0);

add_action('smarty/after_call', function ($smarty) {
    $smarty->registered_plugins['modifier']['cleanHtml'][0] = 'smartyCleanHtml';
}, 0);

function wp_register_style($handle, $src, $deps = array(), $ver = false, $media = 'all')
{
    if (!isset(Helper::$styles[$handle])) {
        Helper::$styles[$handle] = array(
            'src' => $src,
            'deps' => $deps,
            'ver' => $ver,
            'media' => $media,
        );
    }
    return true;
}

function wp_enqueue_style($handle, $src = '', $deps = array(), $ver = false, $media = 'all')
{
    empty($src) or wp_register_style($handle, $src, $deps, $ver, $media);

    if (!empty(Helper::$styles[$handle]) && empty(Helper::$head_styles[$handle])) {
        foreach (Helper::$styles[$handle]['deps'] as $dep) {
            wp_enqueue_style($dep);
        }

        Helper::$head_styles[$handle] = &Helper::$styles[$handle];
        unset(Helper::$styles[$handle]);
    }
}

function wp_register_script($handle, $src, $deps = array(), $ver = false, $in_footer = false)
{
    if (!isset(Helper::$scripts[$handle])) {
        Helper::$scripts[$handle] = array(
            'src' => $src,
            'deps' => $deps,
            'ver' => $ver,
            'head' => !$in_footer,
            'l10n' => array(),
        );
    }
    return true;
}

function wp_enqueue_script($handle, $src = '', $deps = array(), $ver = false, $in_footer = false)
{
    empty($src) or wp_register_script($handle, $src, $deps, $ver, $in_footer);

    if (!empty(Helper::$scripts[$handle]) && empty(Helper::$head_scripts[$handle]) && empty(Helper::$body_scripts[$handle])) {
        foreach (Helper::$scripts[$handle]['deps'] as $dep) {
            wp_enqueue_script($dep);
        }

        if (Helper::$scripts[$handle]['head']) {
            Helper::$head_scripts[$handle] = &Helper::$scripts[$handle];
        } else {
            Helper::$body_scripts[$handle] = &Helper::$scripts[$handle];
        }
        unset(Helper::$scripts[$handle]);
    }
}

function wp_localize_script($handle, $object_name, $l10n)
{
    if (isset(Helper::$scripts[$handle])) {
        Helper::$scripts[$handle]['l10n'][$object_name] = $l10n;
    } elseif (isset(Helper::$head_scripts[$handle])) {
        Helper::$head_scripts[$handle]['l10n'][$object_name] = $l10n;
    } elseif (isset(Helper::$body_scripts[$handle])) {
        Helper::$body_scripts[$handle]['l10n'][$object_name] = $l10n;
    }
}

function wp_enqueue_scripts()
{
    wp_enqueue_script('jquery', _PS_JS_DIR_ . 'jquery/jquery-' . _PS_JQUERY_VERSION_ . '.min.js');

    if (version_compare(_PS_JQUERY_VERSION_, '3.0', '>=')) {
        wp_enqueue_script('jquery-migrate', _CE_ASSETS_URL_ . 'lib/jquery/jquery-migrate.min.js', array('jquery'), '3.3.1');
    }
    wp_enqueue_script('jquery-ui', _CE_ASSETS_URL_ . 'lib/jquery/jquery-ui.min.js', array('jquery'), '1.11.4', true);

    wp_register_script('underscore', _CE_ASSETS_URL_ . 'lib/underscore/underscore.min.js', array(), '1.8.3', true);
    wp_register_script('backbone', _CE_ASSETS_URL_ . 'lib/backbone/backbone.min.js', array('jquery', 'underscore'), '1.2.3', true);

    do_action('wp_enqueue_scripts');
}

function wp_print_styles()
{
    $stylesheet = \CESmarty::get(_CE_TEMPLATES_ . 'admin/admin.tpl', 'ce_stylesheet');

    while ($args = array_shift(Helper::$head_styles)) {
        if ($args['ver']) {
            $args['src'] .= (\Tools::strpos($args['src'], '?') === false ? '?' : '&') . 'v=' . $args['ver'];
        }
        printf($stylesheet, esc_attr($args['src']), esc_attr($args['media']));
    }
}

function wp_print_head_scripts()
{
    while ($args = array_shift(Helper::$head_scripts)) {
        _print_script($args);
    }
}

function wp_print_footer_scripts()
{
    while ($args = array_shift(Helper::$body_scripts)) {
        _print_script($args);
    }
}

function _print_script(&$args)
{
    static $script, $inline_script;

    if (null === $script) {
        $script = \CESmarty::get(_CE_TEMPLATES_ . 'admin/admin.tpl', 'ce_script');
        $inline_script = \CESmarty::get(_CE_TEMPLATES_ . 'admin/admin.tpl', 'ce_inline_script');
    }
    if (!empty($args['l10n'])) {
        $js = '';

        foreach ($args['l10n'] as $key => &$value) {
            $json = json_encode($value);

            if ('ElementorConfig' === $key) {
                // fix for line too long
                $js .= "var $key = " . str_replace('}},"', "}},\n\"", $json) . ";\n";
            } else {
                $js .= "var $key = $json;\n";
            }
        }
        printf($inline_script, $js);
    }
    if (!empty($args['src'])) {
        if (!empty($args['ver'])) {
            $args['src'] .= (\Tools::strpos($args['src'], '?') === false ? '?' : '&') . 'v=' . $args['ver'];
        }
        printf($script, esc_attr($args['src']));
    }
}

function set_transient($transient, $value, $expiration = 0)
{
    $expiration = (int) $expiration;
    $tr_timeout = '_tr_to_' . $transient;
    $tr_option = '_tr_' . $transient;
    $id_shop = \Context::getContext()->shop->id;

    if (false === get_post_meta($id_shop, $tr_option, true)) {
        if ($expiration) {
            update_post_meta($id_shop, $tr_timeout, time() + $expiration);
        }
        $result = update_post_meta($id_shop, $tr_option, $value);
    } else {
        $update = true;
        if ($expiration) {
            if (false === get_post_meta($id_shop, $tr_timeout, true)) {
                update_post_meta($id_shop, $tr_timeout, time() + $expiration);
                $result = update_post_meta($id_shop, $tr_option, $value);
                $update = false;
            } else {
                update_post_meta($id_shop, $tr_timeout, time() + $expiration);
            }
        }
        if ($update) {
            $result = update_post_meta($id_shop, $tr_option, $value);
        }
    }

    return $result;
}

function get_transient($transient)
{
    $tr_option = '_tr_' . $transient;
    $tr_timeout = '_tr_to_' . $transient;
    $id_shop = \Context::getContext()->shop->id;
    $timeout = get_post_meta($id_shop, $tr_timeout, true);

    if (false !== $timeout && $timeout < time()) {
        delete_option($tr_option);
        delete_option($tr_timeout);
        return false;
    }
    return get_post_meta($id_shop, $tr_option, true);
}

define('_CE_ENGLISH_', \Tools::getIsset('en'));

function _e($text, $domain = 'elementor')
{
    echo translate($text, $domain);
}

function __($text, $domain = 'elementor')
{
    return translate($text);
}

function _x($text, $ctx, $domain = 'elementor')
{
    return translate($text, $domain, $ctx);
}

function translate($text, $domain = 'elementor', $ctx = '')
{
    $mod = 'elementor' === $domain ? 'creativeelements' : $domain;
    $src = $ctx ? str_replace(' ', '_', \Tools::strtolower($ctx)) : '';

    return _CE_ENGLISH_ ? $text : call_user_func('strip' . 'slashes', \Translate::getModuleTranslation($mod, $text, $src, null, true));
}

function esc_attr_e($text, $domain = 'elementor')
{
    $mod = 'elementor' === $domain ? 'creativeelements' : $domain;

    echo _CE_ENGLISH_ ? \Tools::safeOutput($text) : \Translate::getModuleTranslation($mod, $text, '');
}

function esc_attr($text)
{
    return \Tools::safeOutput($text);
}

function wp_parse_args($args, $defaults = '')
{
    if (is_object($args)) {
        $r = get_object_vars($args);
    } elseif (is_array($args)) {
        $r = &$args;
    } else {
        parse_str($args, $r);
    }

    if (is_array($defaults)) {
        return array_merge($defaults, $r);
    }
    return $r;
}

function map_deep($value, $callback)
{
    if (is_array($value)) {
        foreach ($value as $index => $item) {
            $value[$index] = map_deep($item, $callback);
        }
    } elseif (is_object($value)) {
        $object_vars = get_object_vars($value);
        foreach ($object_vars as $property_name => $property_value) {
            $value->$property_name = map_deep($property_value, $callback);
        }
    } else {
        $value = call_user_func($callback, $value);
    }
    return $value;
}

function esc_url($url, $protocols = null, $_context = 'display')
{
    if ('' == $url) {
        return $url;
    }
    $url = str_replace(' ', '%20', $url);
    $url = preg_replace('|[^a-z0-9-~+_.?#=!&;,/:%@$\|*\'()\[\]\\x80-\\xff]|i', '', $url);
    if ('' === $url) {
        return $url;
    }
    $url = str_replace(';//', '://', $url);
    if (strpos($url, ':') === false && !in_array($url[0], array('/', '#', '?')) &&
        !preg_match('/^[a-z0-9-]+?\.php/i', $url)) {
        $url = 'http://' . $url;
    }
    return $url;
}

function wp_send_json($response/*, $status_code = null*/)
{
    headers_sent() or header('Content-Type: application/json; charset=utf-8');

    die(json_encode($response));
}

function wp_send_json_success($data = null)
{
    $response = array('success' => true);
    if (isset($data)) {
        $response['data'] = $data;
    }
    wp_send_json($response);
}

function wp_send_json_error($data = null)
{
    $response = array('success' => false);
    if (isset($data)) {
        $response['data'] = $data;
    }
    wp_send_json($response);
}

function is_rtl()
{
    return !empty(\Context::getContext()->language->is_rtl);
}

function is_admin()
{
    return defined('_PS_ADMIN_DIR_') && _PS_ADMIN_DIR_;
}

function get_option($option, $default = false)
{
    if (false === $res = \Configuration::get($option)) {
        return $default;
    }
    return isset($res[0]) && ('{' == $res[0] || '[' == $res[0]) ? json_decode($res, true) : $res;
}

function update_option($option, $value)
{
    if (is_array($value) || is_object($value)) {
        $value = json_encode($value);
    }
    $purify = \Configuration::get('PS_USE_HTMLPURIFIER');
    empty($purify) or \Configuration::set('PS_USE_HTMLPURIFIER', 0);

    $res = \Configuration::updateValue($option, array($value), true);

    if (\Shop::CONTEXT_SHOP !== $shop_ctx = \Shop::getContext()) {
        $groups = \Shop::CONTEXT_ALL === $shop_ctx ? new \stdClass() : false;

        foreach (\Shop::getContextListShopID() as $id_shop) {
            $id_shop_group = \Shop::getGroupFromShop($id_shop);

            if ($groups && empty($groups->$id_shop_group)) {
                $groups->$id_shop_group = true;

                $res &= \Configuration::updateValue($option, array($value), true, $id_shop_group);
            }
            $res &= \Configuration::updateValue($option, array($value), true, $id_shop_group, $id_shop);
        }
    }
    empty($purify) or \Configuration::set('PS_USE_HTMLPURIFIER', 1);

    return $res;
}

function delete_option($option)
{
    return \Configuration::deleteByName($option);
}

function get_current_user_id()
{
    $ctx = \Context::getContext();
    return empty($ctx->employee->id) ? 0 : (int) $ctx->employee->id;
}

function wp_get_current_user()
{
    $user = \Context::getContext()->employee;
    return (object) array(
        'ID' => $user->id,
        'display_name' => $user->firstname . ' ' . $user->lastname,
        'roles' => array(),
    );
}

function get_user_by($field, $value)
{
    if ('id' != $field) {
        throw new \RuntimeException('TODO');
    }
    if (!\Validate::isLoadedObject($user = new \Employee($value))) {
        return false;
    }
    return (object) array(
        'ID' => $user->id,
        'display_name' => $user->firstname . ' ' . $user->lastname,
        'roles' => array(),
    );
}

function wp_set_post_lock($post_id)
{
    if (!$user_id = get_current_user_id()) {
        return false;
    }
    $now = time();

    update_post_meta($post_id, '_edit_lock', "$now:$user_id");

    return array($now, $user_id);
}

function wp_check_post_lock($post_id)
{
    if (!$lock = get_post_meta($post_id, '_edit_lock', true)) {
        return false;
    }
    list($time, $user) = explode(':', $lock);

    if (empty($user)) {
        return false;
    }
    $time_window = apply_filters('wp_check_post_lock_window', 150);

    if ($time && $time > time() - $time_window && $user != get_current_user_id()) {
        return (int) $user;
    }
    return false;
}

function wp_remote_post($url, array $args = array())
{
    $http = array(
        'method' => 'POST',
        'header' => 'Content-type: application/x-www-form-urlencoded',
        'user_agent' => $_SERVER['SERVER_SOFTWARE'],
        'content' => empty($args['body']) ? '' : http_build_query($args['body']),
        'max_redirects' => 5,
        'timeout' => empty($args['timeout']) ? 5 : $args['timeout'],
    );

    if (ini_get('allow_url_fopen')) {
        return \Tools::file_get_contents($url, false, stream_context_create(array('http' => $http)), $http['timeout']);
    }

    $ch = curl_init($url);
    curl_setopt_array($ch, array(
        CURLOPT_POST => 1,
        CURLOPT_HTTPHEADER => (array) $http['header'],
        CURLOPT_USERAGENT => $http['user_agent'],
        CURLOPT_POSTFIELDS => $http['content'],
        CURLOPT_MAXREDIRS => $http['max_redirects'],
        CURLOPT_TIMEOUT => $http['timeout'],
        CURLOPT_RETURNTRANSFER => 1,
    ));
    $resp = curl_exec($ch);
    curl_close($ch);

    return $resp;
}

function wp_remote_get($url, array $args = array())
{
    $http = array(
        'method' => 'GET',
        'user_agent' => $_SERVER['SERVER_SOFTWARE'],
        'max_redirects' => 5,
        'timeout' => empty($args['timeout']) ? 5 : $args['timeout'],
    );

    if (!empty($args['body'])) {
        $url .= (strpos($url, '?') === false ? '?' : '&') . http_build_query($args['body']);
    }

    if (ini_get('allow_url_fopen')) {
        return \Tools::file_get_contents($url, false, stream_context_create(array('http' => $http)), $http['timeout']);
    }

    $ch = curl_init($url);
    curl_setopt_array($ch, array(
        CURLOPT_USERAGENT => $http['user_agent'],
        CURLOPT_MAXREDIRS => $http['max_redirects'],
        CURLOPT_TIMEOUT => $http['timeout'],
        CURLOPT_RETURNTRANSFER => 1,
    ));
    $resp = curl_exec($ch);
    curl_close($ch);

    return $resp;
}

function _doing_it_wrong($function, $message = '', $version = '')
{
    die(\Tools::displayError($function . ' was called incorrectly. ' . $message . ' ' . $version));
}

function is_wp_error($error)
{
    return $error instanceof \PrestaShopException;
}
